import { r as resolveUrl, f as forEachMediaGroup, d as decodeB64ToUint8Array } from "./@videojs.e887769e.js";
import { w as window_1 } from "./global.dfc622c4.js";
import { D as DOMParser } from "./@xmldom.63d5d2ec.js";
/*! @name mpd-parser @version 1.3.1 @license Apache-2.0 */
const isObject = (obj) => {
  return !!obj && typeof obj === "object";
};
const merge = (...objects) => {
  return objects.reduce((result, source) => {
    if (typeof source !== "object") {
      return result;
    }
    Object.keys(source).forEach((key) => {
      if (Array.isArray(result[key]) && Array.isArray(source[key])) {
        result[key] = result[key].concat(source[key]);
      } else if (isObject(result[key]) && isObject(source[key])) {
        result[key] = merge(result[key], source[key]);
      } else {
        result[key] = source[key];
      }
    });
    return result;
  }, {});
};
const values = (o) => Object.keys(o).map((k) => o[k]);
const range = (start, end) => {
  const result = [];
  for (let i = start; i < end; i++) {
    result.push(i);
  }
  return result;
};
const flatten = (lists) => lists.reduce((x, y) => x.concat(y), []);
const from = (list) => {
  if (!list.length) {
    return [];
  }
  const result = [];
  for (let i = 0; i < list.length; i++) {
    result.push(list[i]);
  }
  return result;
};
const findIndexes = (l, key) => l.reduce((a, e, i) => {
  if (e[key]) {
    a.push(i);
  }
  return a;
}, []);
const union = (lists, keyFunction) => {
  return values(lists.reduce((acc, list) => {
    list.forEach((el) => {
      acc[keyFunction(el)] = el;
    });
    return acc;
  }, {}));
};
var errors = {
  INVALID_NUMBER_OF_PERIOD: "INVALID_NUMBER_OF_PERIOD",
  INVALID_NUMBER_OF_CONTENT_STEERING: "INVALID_NUMBER_OF_CONTENT_STEERING",
  DASH_EMPTY_MANIFEST: "DASH_EMPTY_MANIFEST",
  DASH_INVALID_XML: "DASH_INVALID_XML",
  NO_BASE_URL: "NO_BASE_URL",
  MISSING_SEGMENT_INFORMATION: "MISSING_SEGMENT_INFORMATION",
  SEGMENT_TIME_UNSPECIFIED: "SEGMENT_TIME_UNSPECIFIED",
  UNSUPPORTED_UTC_TIMING_SCHEME: "UNSUPPORTED_UTC_TIMING_SCHEME"
};
const urlTypeToSegment = ({
  baseUrl = "",
  source = "",
  range: range2 = "",
  indexRange = ""
}) => {
  const segment = {
    uri: source,
    resolvedUri: resolveUrl(baseUrl || "", source)
  };
  if (range2 || indexRange) {
    const rangeStr = range2 ? range2 : indexRange;
    const ranges = rangeStr.split("-");
    let startRange = window_1.BigInt ? window_1.BigInt(ranges[0]) : parseInt(ranges[0], 10);
    let endRange = window_1.BigInt ? window_1.BigInt(ranges[1]) : parseInt(ranges[1], 10);
    if (startRange < Number.MAX_SAFE_INTEGER && typeof startRange === "bigint") {
      startRange = Number(startRange);
    }
    if (endRange < Number.MAX_SAFE_INTEGER && typeof endRange === "bigint") {
      endRange = Number(endRange);
    }
    let length;
    if (typeof endRange === "bigint" || typeof startRange === "bigint") {
      length = window_1.BigInt(endRange) - window_1.BigInt(startRange) + window_1.BigInt(1);
    } else {
      length = endRange - startRange + 1;
    }
    if (typeof length === "bigint" && length < Number.MAX_SAFE_INTEGER) {
      length = Number(length);
    }
    segment.byterange = {
      length,
      offset: startRange
    };
  }
  return segment;
};
const byteRangeToString = (byterange) => {
  let endRange;
  if (typeof byterange.offset === "bigint" || typeof byterange.length === "bigint") {
    endRange = window_1.BigInt(byterange.offset) + window_1.BigInt(byterange.length) - window_1.BigInt(1);
  } else {
    endRange = byterange.offset + byterange.length - 1;
  }
  return `${byterange.offset}-${endRange}`;
};
const parseEndNumber = (endNumber) => {
  if (endNumber && typeof endNumber !== "number") {
    endNumber = parseInt(endNumber, 10);
  }
  if (isNaN(endNumber)) {
    return null;
  }
  return endNumber;
};
const segmentRange = {
  static(attributes) {
    const {
      duration,
      timescale = 1,
      sourceDuration,
      periodDuration
    } = attributes;
    const endNumber = parseEndNumber(attributes.endNumber);
    const segmentDuration = duration / timescale;
    if (typeof endNumber === "number") {
      return {
        start: 0,
        end: endNumber
      };
    }
    if (typeof periodDuration === "number") {
      return {
        start: 0,
        end: periodDuration / segmentDuration
      };
    }
    return {
      start: 0,
      end: sourceDuration / segmentDuration
    };
  },
  dynamic(attributes) {
    const {
      NOW,
      clientOffset,
      availabilityStartTime,
      timescale = 1,
      duration,
      periodStart = 0,
      minimumUpdatePeriod = 0,
      timeShiftBufferDepth = Infinity
    } = attributes;
    const endNumber = parseEndNumber(attributes.endNumber);
    const now = (NOW + clientOffset) / 1e3;
    const periodStartWC = availabilityStartTime + periodStart;
    const periodEndWC = now + minimumUpdatePeriod;
    const periodDuration = periodEndWC - periodStartWC;
    const segmentCount = Math.ceil(periodDuration * timescale / duration);
    const availableStart = Math.floor((now - periodStartWC - timeShiftBufferDepth) * timescale / duration);
    const availableEnd = Math.floor((now - periodStartWC) * timescale / duration);
    return {
      start: Math.max(0, availableStart),
      end: typeof endNumber === "number" ? endNumber : Math.min(segmentCount, availableEnd)
    };
  }
};
const toSegments = (attributes) => (number) => {
  const {
    duration,
    timescale = 1,
    periodStart,
    startNumber = 1
  } = attributes;
  return {
    number: startNumber + number,
    duration: duration / timescale,
    timeline: periodStart,
    time: number * duration
  };
};
const parseByDuration = (attributes) => {
  const {
    type,
    duration,
    timescale = 1,
    periodDuration,
    sourceDuration
  } = attributes;
  const {
    start,
    end
  } = segmentRange[type](attributes);
  const segments = range(start, end).map(toSegments(attributes));
  if (type === "static") {
    const index = segments.length - 1;
    const sectionDuration = typeof periodDuration === "number" ? periodDuration : sourceDuration;
    segments[index].duration = sectionDuration - duration / timescale * index;
  }
  return segments;
};
const segmentsFromBase = (attributes) => {
  const {
    baseUrl,
    initialization = {},
    sourceDuration,
    indexRange = "",
    periodStart,
    presentationTime,
    number = 0,
    duration
  } = attributes;
  if (!baseUrl) {
    throw new Error(errors.NO_BASE_URL);
  }
  const initSegment = urlTypeToSegment({
    baseUrl,
    source: initialization.sourceURL,
    range: initialization.range
  });
  const segment = urlTypeToSegment({
    baseUrl,
    source: baseUrl,
    indexRange
  });
  segment.map = initSegment;
  if (duration) {
    const segmentTimeInfo = parseByDuration(attributes);
    if (segmentTimeInfo.length) {
      segment.duration = segmentTimeInfo[0].duration;
      segment.timeline = segmentTimeInfo[0].timeline;
    }
  } else if (sourceDuration) {
    segment.duration = sourceDuration;
    segment.timeline = periodStart;
  }
  segment.presentationTime = presentationTime || periodStart;
  segment.number = number;
  return [segment];
};
const addSidxSegmentsToPlaylist$1 = (playlist, sidx, baseUrl) => {
  const initSegment = playlist.sidx.map ? playlist.sidx.map : null;
  const sourceDuration = playlist.sidx.duration;
  const timeline = playlist.timeline || 0;
  const sidxByteRange = playlist.sidx.byterange;
  const sidxEnd = sidxByteRange.offset + sidxByteRange.length;
  const timescale = sidx.timescale;
  const mediaReferences = sidx.references.filter((r) => r.referenceType !== 1);
  const segments = [];
  const type = playlist.endList ? "static" : "dynamic";
  const periodStart = playlist.sidx.timeline;
  let presentationTime = periodStart;
  let number = playlist.mediaSequence || 0;
  let startIndex;
  if (typeof sidx.firstOffset === "bigint") {
    startIndex = window_1.BigInt(sidxEnd) + sidx.firstOffset;
  } else {
    startIndex = sidxEnd + sidx.firstOffset;
  }
  for (let i = 0; i < mediaReferences.length; i++) {
    const reference = sidx.references[i];
    const size = reference.referencedSize;
    const duration = reference.subsegmentDuration;
    let endIndex;
    if (typeof startIndex === "bigint") {
      endIndex = startIndex + window_1.BigInt(size) - window_1.BigInt(1);
    } else {
      endIndex = startIndex + size - 1;
    }
    const indexRange = `${startIndex}-${endIndex}`;
    const attributes = {
      baseUrl,
      timescale,
      timeline,
      periodStart,
      presentationTime,
      number,
      duration,
      sourceDuration,
      indexRange,
      type
    };
    const segment = segmentsFromBase(attributes)[0];
    if (initSegment) {
      segment.map = initSegment;
    }
    segments.push(segment);
    if (typeof startIndex === "bigint") {
      startIndex += window_1.BigInt(size);
    } else {
      startIndex += size;
    }
    presentationTime += duration / timescale;
    number++;
  }
  playlist.segments = segments;
  return playlist;
};
const SUPPORTED_MEDIA_TYPES = ["AUDIO", "SUBTITLES"];
const TIME_FUDGE = 1 / 60;
const getUniqueTimelineStarts = (timelineStarts) => {
  return union(timelineStarts, ({
    timeline
  }) => timeline).sort((a, b) => a.timeline > b.timeline ? 1 : -1);
};
const findPlaylistWithName = (playlists, name) => {
  for (let i = 0; i < playlists.length; i++) {
    if (playlists[i].attributes.NAME === name) {
      return playlists[i];
    }
  }
  return null;
};
const getMediaGroupPlaylists = (manifest) => {
  let mediaGroupPlaylists = [];
  forEachMediaGroup(manifest, SUPPORTED_MEDIA_TYPES, (properties, type, group, label) => {
    mediaGroupPlaylists = mediaGroupPlaylists.concat(properties.playlists || []);
  });
  return mediaGroupPlaylists;
};
const updateMediaSequenceForPlaylist = ({
  playlist,
  mediaSequence
}) => {
  playlist.mediaSequence = mediaSequence;
  playlist.segments.forEach((segment, index) => {
    segment.number = playlist.mediaSequence + index;
  });
};
const updateSequenceNumbers = ({
  oldPlaylists,
  newPlaylists,
  timelineStarts
}) => {
  newPlaylists.forEach((playlist) => {
    playlist.discontinuitySequence = timelineStarts.findIndex(function({
      timeline
    }) {
      return timeline === playlist.timeline;
    });
    const oldPlaylist = findPlaylistWithName(oldPlaylists, playlist.attributes.NAME);
    if (!oldPlaylist) {
      return;
    }
    if (playlist.sidx) {
      return;
    }
    const firstNewSegment = playlist.segments[0];
    const oldMatchingSegmentIndex = oldPlaylist.segments.findIndex(function(oldSegment) {
      return Math.abs(oldSegment.presentationTime - firstNewSegment.presentationTime) < TIME_FUDGE;
    });
    if (oldMatchingSegmentIndex === -1) {
      updateMediaSequenceForPlaylist({
        playlist,
        mediaSequence: oldPlaylist.mediaSequence + oldPlaylist.segments.length
      });
      playlist.segments[0].discontinuity = true;
      playlist.discontinuityStarts.unshift(0);
      if (!oldPlaylist.segments.length && playlist.timeline > oldPlaylist.timeline || oldPlaylist.segments.length && playlist.timeline > oldPlaylist.segments[oldPlaylist.segments.length - 1].timeline) {
        playlist.discontinuitySequence--;
      }
      return;
    }
    const oldMatchingSegment = oldPlaylist.segments[oldMatchingSegmentIndex];
    if (oldMatchingSegment.discontinuity && !firstNewSegment.discontinuity) {
      firstNewSegment.discontinuity = true;
      playlist.discontinuityStarts.unshift(0);
      playlist.discontinuitySequence--;
    }
    updateMediaSequenceForPlaylist({
      playlist,
      mediaSequence: oldPlaylist.segments[oldMatchingSegmentIndex].number
    });
  });
};
const positionManifestOnTimeline = ({
  oldManifest,
  newManifest
}) => {
  const oldPlaylists = oldManifest.playlists.concat(getMediaGroupPlaylists(oldManifest));
  const newPlaylists = newManifest.playlists.concat(getMediaGroupPlaylists(newManifest));
  newManifest.timelineStarts = getUniqueTimelineStarts([oldManifest.timelineStarts, newManifest.timelineStarts]);
  updateSequenceNumbers({
    oldPlaylists,
    newPlaylists,
    timelineStarts: newManifest.timelineStarts
  });
  return newManifest;
};
const generateSidxKey = (sidx) => sidx && sidx.uri + "-" + byteRangeToString(sidx.byterange);
const mergeDiscontiguousPlaylists = (playlists) => {
  const playlistsByBaseUrl = playlists.reduce(function(acc, cur) {
    if (!acc[cur.attributes.baseUrl]) {
      acc[cur.attributes.baseUrl] = [];
    }
    acc[cur.attributes.baseUrl].push(cur);
    return acc;
  }, {});
  let allPlaylists = [];
  Object.values(playlistsByBaseUrl).forEach((playlistGroup) => {
    const mergedPlaylists = values(playlistGroup.reduce((acc, playlist) => {
      const name = playlist.attributes.id + (playlist.attributes.lang || "");
      if (!acc[name]) {
        acc[name] = playlist;
        acc[name].attributes.timelineStarts = [];
      } else {
        if (playlist.segments) {
          if (playlist.segments[0]) {
            playlist.segments[0].discontinuity = true;
          }
          acc[name].segments.push(...playlist.segments);
        }
        if (playlist.attributes.contentProtection) {
          acc[name].attributes.contentProtection = playlist.attributes.contentProtection;
        }
      }
      acc[name].attributes.timelineStarts.push({
        start: playlist.attributes.periodStart,
        timeline: playlist.attributes.periodStart
      });
      return acc;
    }, {}));
    allPlaylists = allPlaylists.concat(mergedPlaylists);
  });
  return allPlaylists.map((playlist) => {
    playlist.discontinuityStarts = findIndexes(playlist.segments || [], "discontinuity");
    return playlist;
  });
};
const addSidxSegmentsToPlaylist = (playlist, sidxMapping) => {
  const sidxKey = generateSidxKey(playlist.sidx);
  const sidxMatch = sidxKey && sidxMapping[sidxKey] && sidxMapping[sidxKey].sidx;
  if (sidxMatch) {
    addSidxSegmentsToPlaylist$1(playlist, sidxMatch, playlist.sidx.resolvedUri);
  }
  return playlist;
};
const addSidxSegmentsToPlaylists = (playlists, sidxMapping = {}) => {
  if (!Object.keys(sidxMapping).length) {
    return playlists;
  }
  for (const i in playlists) {
    playlists[i] = addSidxSegmentsToPlaylist(playlists[i], sidxMapping);
  }
  return playlists;
};
const formatAudioPlaylist = ({
  attributes,
  segments,
  sidx,
  mediaSequence,
  discontinuitySequence,
  discontinuityStarts
}, isAudioOnly) => {
  const playlist = {
    attributes: {
      NAME: attributes.id,
      BANDWIDTH: attributes.bandwidth,
      CODECS: attributes.codecs,
      ["PROGRAM-ID"]: 1
    },
    uri: "",
    endList: attributes.type === "static",
    timeline: attributes.periodStart,
    resolvedUri: attributes.baseUrl || "",
    targetDuration: attributes.duration,
    discontinuitySequence,
    discontinuityStarts,
    timelineStarts: attributes.timelineStarts,
    mediaSequence,
    segments
  };
  if (attributes.contentProtection) {
    playlist.contentProtection = attributes.contentProtection;
  }
  if (attributes.serviceLocation) {
    playlist.attributes.serviceLocation = attributes.serviceLocation;
  }
  if (sidx) {
    playlist.sidx = sidx;
  }
  if (isAudioOnly) {
    playlist.attributes.AUDIO = "audio";
    playlist.attributes.SUBTITLES = "subs";
  }
  return playlist;
};
const formatVttPlaylist = ({
  attributes,
  segments,
  mediaSequence,
  discontinuityStarts,
  discontinuitySequence
}) => {
  if (typeof segments === "undefined") {
    segments = [{
      uri: attributes.baseUrl,
      timeline: attributes.periodStart,
      resolvedUri: attributes.baseUrl || "",
      duration: attributes.sourceDuration,
      number: 0
    }];
    attributes.duration = attributes.sourceDuration;
  }
  const m3u8Attributes = {
    NAME: attributes.id,
    BANDWIDTH: attributes.bandwidth,
    ["PROGRAM-ID"]: 1
  };
  if (attributes.codecs) {
    m3u8Attributes.CODECS = attributes.codecs;
  }
  const vttPlaylist = {
    attributes: m3u8Attributes,
    uri: "",
    endList: attributes.type === "static",
    timeline: attributes.periodStart,
    resolvedUri: attributes.baseUrl || "",
    targetDuration: attributes.duration,
    timelineStarts: attributes.timelineStarts,
    discontinuityStarts,
    discontinuitySequence,
    mediaSequence,
    segments
  };
  if (attributes.serviceLocation) {
    vttPlaylist.attributes.serviceLocation = attributes.serviceLocation;
  }
  return vttPlaylist;
};
const organizeAudioPlaylists = (playlists, sidxMapping = {}, isAudioOnly = false) => {
  let mainPlaylist;
  const formattedPlaylists = playlists.reduce((a, playlist) => {
    const role = playlist.attributes.role && playlist.attributes.role.value || "";
    const language = playlist.attributes.lang || "";
    let label = playlist.attributes.label || "main";
    if (language && !playlist.attributes.label) {
      const roleLabel = role ? ` (${role})` : "";
      label = `${playlist.attributes.lang}${roleLabel}`;
    }
    if (!a[label]) {
      a[label] = {
        language,
        autoselect: true,
        default: role === "main",
        playlists: [],
        uri: ""
      };
    }
    const formatted = addSidxSegmentsToPlaylist(formatAudioPlaylist(playlist, isAudioOnly), sidxMapping);
    a[label].playlists.push(formatted);
    if (typeof mainPlaylist === "undefined" && role === "main") {
      mainPlaylist = playlist;
      mainPlaylist.default = true;
    }
    return a;
  }, {});
  if (!mainPlaylist) {
    const firstLabel = Object.keys(formattedPlaylists)[0];
    formattedPlaylists[firstLabel].default = true;
  }
  return formattedPlaylists;
};
const organizeVttPlaylists = (playlists, sidxMapping = {}) => {
  return playlists.reduce((a, playlist) => {
    const label = playlist.attributes.label || playlist.attributes.lang || "text";
    const language = playlist.attributes.lang || "und";
    if (!a[label]) {
      a[label] = {
        language,
        default: false,
        autoselect: false,
        playlists: [],
        uri: ""
      };
    }
    a[label].playlists.push(addSidxSegmentsToPlaylist(formatVttPlaylist(playlist), sidxMapping));
    return a;
  }, {});
};
const organizeCaptionServices = (captionServices) => captionServices.reduce((svcObj, svc) => {
  if (!svc) {
    return svcObj;
  }
  svc.forEach((service) => {
    const {
      channel,
      language
    } = service;
    svcObj[language] = {
      autoselect: false,
      default: false,
      instreamId: channel,
      language
    };
    if (service.hasOwnProperty("aspectRatio")) {
      svcObj[language].aspectRatio = service.aspectRatio;
    }
    if (service.hasOwnProperty("easyReader")) {
      svcObj[language].easyReader = service.easyReader;
    }
    if (service.hasOwnProperty("3D")) {
      svcObj[language]["3D"] = service["3D"];
    }
  });
  return svcObj;
}, {});
const formatVideoPlaylist = ({
  attributes,
  segments,
  sidx,
  discontinuityStarts
}) => {
  const playlist = {
    attributes: {
      NAME: attributes.id,
      AUDIO: "audio",
      SUBTITLES: "subs",
      RESOLUTION: {
        width: attributes.width,
        height: attributes.height
      },
      CODECS: attributes.codecs,
      BANDWIDTH: attributes.bandwidth,
      ["PROGRAM-ID"]: 1
    },
    uri: "",
    endList: attributes.type === "static",
    timeline: attributes.periodStart,
    resolvedUri: attributes.baseUrl || "",
    targetDuration: attributes.duration,
    discontinuityStarts,
    timelineStarts: attributes.timelineStarts,
    segments
  };
  if (attributes.frameRate) {
    playlist.attributes["FRAME-RATE"] = attributes.frameRate;
  }
  if (attributes.contentProtection) {
    playlist.contentProtection = attributes.contentProtection;
  }
  if (attributes.serviceLocation) {
    playlist.attributes.serviceLocation = attributes.serviceLocation;
  }
  if (sidx) {
    playlist.sidx = sidx;
  }
  return playlist;
};
const videoOnly = ({
  attributes
}) => attributes.mimeType === "video/mp4" || attributes.mimeType === "video/webm" || attributes.contentType === "video";
const audioOnly = ({
  attributes
}) => attributes.mimeType === "audio/mp4" || attributes.mimeType === "audio/webm" || attributes.contentType === "audio";
const vttOnly = ({
  attributes
}) => attributes.mimeType === "text/vtt" || attributes.contentType === "text";
const addMediaSequenceValues = (playlists, timelineStarts) => {
  playlists.forEach((playlist) => {
    playlist.mediaSequence = 0;
    playlist.discontinuitySequence = timelineStarts.findIndex(function({
      timeline
    }) {
      return timeline === playlist.timeline;
    });
    if (!playlist.segments) {
      return;
    }
    playlist.segments.forEach((segment, index) => {
      segment.number = index;
    });
  });
};
const flattenMediaGroupPlaylists = (mediaGroupObject) => {
  if (!mediaGroupObject) {
    return [];
  }
  return Object.keys(mediaGroupObject).reduce((acc, label) => {
    const labelContents = mediaGroupObject[label];
    return acc.concat(labelContents.playlists);
  }, []);
};
const toM3u8 = ({
  dashPlaylists,
  locations,
  contentSteering,
  sidxMapping = {},
  previousManifest,
  eventStream
}) => {
  if (!dashPlaylists.length) {
    return {};
  }
  const {
    sourceDuration: duration,
    type,
    suggestedPresentationDelay,
    minimumUpdatePeriod
  } = dashPlaylists[0].attributes;
  const videoPlaylists = mergeDiscontiguousPlaylists(dashPlaylists.filter(videoOnly)).map(formatVideoPlaylist);
  const audioPlaylists = mergeDiscontiguousPlaylists(dashPlaylists.filter(audioOnly));
  const vttPlaylists = mergeDiscontiguousPlaylists(dashPlaylists.filter(vttOnly));
  const captions = dashPlaylists.map((playlist) => playlist.attributes.captionServices).filter(Boolean);
  const manifest = {
    allowCache: true,
    discontinuityStarts: [],
    segments: [],
    endList: true,
    mediaGroups: {
      AUDIO: {},
      VIDEO: {},
      ["CLOSED-CAPTIONS"]: {},
      SUBTITLES: {}
    },
    uri: "",
    duration,
    playlists: addSidxSegmentsToPlaylists(videoPlaylists, sidxMapping)
  };
  if (minimumUpdatePeriod >= 0) {
    manifest.minimumUpdatePeriod = minimumUpdatePeriod * 1e3;
  }
  if (locations) {
    manifest.locations = locations;
  }
  if (contentSteering) {
    manifest.contentSteering = contentSteering;
  }
  if (type === "dynamic") {
    manifest.suggestedPresentationDelay = suggestedPresentationDelay;
  }
  if (eventStream && eventStream.length > 0) {
    manifest.eventStream = eventStream;
  }
  const isAudioOnly = manifest.playlists.length === 0;
  const organizedAudioGroup = audioPlaylists.length ? organizeAudioPlaylists(audioPlaylists, sidxMapping, isAudioOnly) : null;
  const organizedVttGroup = vttPlaylists.length ? organizeVttPlaylists(vttPlaylists, sidxMapping) : null;
  const formattedPlaylists = videoPlaylists.concat(flattenMediaGroupPlaylists(organizedAudioGroup), flattenMediaGroupPlaylists(organizedVttGroup));
  const playlistTimelineStarts = formattedPlaylists.map(({
    timelineStarts
  }) => timelineStarts);
  manifest.timelineStarts = getUniqueTimelineStarts(playlistTimelineStarts);
  addMediaSequenceValues(formattedPlaylists, manifest.timelineStarts);
  if (organizedAudioGroup) {
    manifest.mediaGroups.AUDIO.audio = organizedAudioGroup;
  }
  if (organizedVttGroup) {
    manifest.mediaGroups.SUBTITLES.subs = organizedVttGroup;
  }
  if (captions.length) {
    manifest.mediaGroups["CLOSED-CAPTIONS"].cc = organizeCaptionServices(captions);
  }
  if (previousManifest) {
    return positionManifestOnTimeline({
      oldManifest: previousManifest,
      newManifest: manifest
    });
  }
  return manifest;
};
const getLiveRValue = (attributes, time, duration) => {
  const {
    NOW,
    clientOffset,
    availabilityStartTime,
    timescale = 1,
    periodStart = 0,
    minimumUpdatePeriod = 0
  } = attributes;
  const now = (NOW + clientOffset) / 1e3;
  const periodStartWC = availabilityStartTime + periodStart;
  const periodEndWC = now + minimumUpdatePeriod;
  const periodDuration = periodEndWC - periodStartWC;
  return Math.ceil((periodDuration * timescale - time) / duration);
};
const parseByTimeline = (attributes, segmentTimeline) => {
  const {
    type,
    minimumUpdatePeriod = 0,
    media = "",
    sourceDuration,
    timescale = 1,
    startNumber = 1,
    periodStart: timeline
  } = attributes;
  const segments = [];
  let time = -1;
  for (let sIndex = 0; sIndex < segmentTimeline.length; sIndex++) {
    const S = segmentTimeline[sIndex];
    const duration = S.d;
    const repeat = S.r || 0;
    const segmentTime = S.t || 0;
    if (time < 0) {
      time = segmentTime;
    }
    if (segmentTime && segmentTime > time) {
      time = segmentTime;
    }
    let count;
    if (repeat < 0) {
      const nextS = sIndex + 1;
      if (nextS === segmentTimeline.length) {
        if (type === "dynamic" && minimumUpdatePeriod > 0 && media.indexOf("$Number$") > 0) {
          count = getLiveRValue(attributes, time, duration);
        } else {
          count = (sourceDuration * timescale - time) / duration;
        }
      } else {
        count = (segmentTimeline[nextS].t - time) / duration;
      }
    } else {
      count = repeat + 1;
    }
    const end = startNumber + segments.length + count;
    let number = startNumber + segments.length;
    while (number < end) {
      segments.push({
        number,
        duration: duration / timescale,
        time,
        timeline
      });
      time += duration;
      number++;
    }
  }
  return segments;
};
const identifierPattern = /\$([A-z]*)(?:(%0)([0-9]+)d)?\$/g;
const identifierReplacement = (values2) => (match, identifier, format, width) => {
  if (match === "$$") {
    return "$";
  }
  if (typeof values2[identifier] === "undefined") {
    return match;
  }
  const value = "" + values2[identifier];
  if (identifier === "RepresentationID") {
    return value;
  }
  if (!format) {
    width = 1;
  } else {
    width = parseInt(width, 10);
  }
  if (value.length >= width) {
    return value;
  }
  return `${new Array(width - value.length + 1).join("0")}${value}`;
};
const constructTemplateUrl = (url, values2) => url.replace(identifierPattern, identifierReplacement(values2));
const parseTemplateInfo = (attributes, segmentTimeline) => {
  if (!attributes.duration && !segmentTimeline) {
    return [{
      number: attributes.startNumber || 1,
      duration: attributes.sourceDuration,
      time: 0,
      timeline: attributes.periodStart
    }];
  }
  if (attributes.duration) {
    return parseByDuration(attributes);
  }
  return parseByTimeline(attributes, segmentTimeline);
};
const segmentsFromTemplate = (attributes, segmentTimeline) => {
  const templateValues = {
    RepresentationID: attributes.id,
    Bandwidth: attributes.bandwidth || 0
  };
  const {
    initialization = {
      sourceURL: "",
      range: ""
    }
  } = attributes;
  const mapSegment = urlTypeToSegment({
    baseUrl: attributes.baseUrl,
    source: constructTemplateUrl(initialization.sourceURL, templateValues),
    range: initialization.range
  });
  const segments = parseTemplateInfo(attributes, segmentTimeline);
  return segments.map((segment) => {
    templateValues.Number = segment.number;
    templateValues.Time = segment.time;
    const uri = constructTemplateUrl(attributes.media || "", templateValues);
    const timescale = attributes.timescale || 1;
    const presentationTimeOffset = attributes.presentationTimeOffset || 0;
    const presentationTime = attributes.periodStart + (segment.time - presentationTimeOffset) / timescale;
    const map = {
      uri,
      timeline: segment.timeline,
      duration: segment.duration,
      resolvedUri: resolveUrl(attributes.baseUrl || "", uri),
      map: mapSegment,
      number: segment.number,
      presentationTime
    };
    return map;
  });
};
const SegmentURLToSegmentObject = (attributes, segmentUrl) => {
  const {
    baseUrl,
    initialization = {}
  } = attributes;
  const initSegment = urlTypeToSegment({
    baseUrl,
    source: initialization.sourceURL,
    range: initialization.range
  });
  const segment = urlTypeToSegment({
    baseUrl,
    source: segmentUrl.media,
    range: segmentUrl.mediaRange
  });
  segment.map = initSegment;
  return segment;
};
const segmentsFromList = (attributes, segmentTimeline) => {
  const {
    duration,
    segmentUrls = [],
    periodStart
  } = attributes;
  if (!duration && !segmentTimeline || duration && segmentTimeline) {
    throw new Error(errors.SEGMENT_TIME_UNSPECIFIED);
  }
  const segmentUrlMap = segmentUrls.map((segmentUrlObject) => SegmentURLToSegmentObject(attributes, segmentUrlObject));
  let segmentTimeInfo;
  if (duration) {
    segmentTimeInfo = parseByDuration(attributes);
  }
  if (segmentTimeline) {
    segmentTimeInfo = parseByTimeline(attributes, segmentTimeline);
  }
  const segments = segmentTimeInfo.map((segmentTime, index) => {
    if (segmentUrlMap[index]) {
      const segment = segmentUrlMap[index];
      const timescale = attributes.timescale || 1;
      const presentationTimeOffset = attributes.presentationTimeOffset || 0;
      segment.timeline = segmentTime.timeline;
      segment.duration = segmentTime.duration;
      segment.number = segmentTime.number;
      segment.presentationTime = periodStart + (segmentTime.time - presentationTimeOffset) / timescale;
      return segment;
    }
  }).filter((segment) => segment);
  return segments;
};
const generateSegments = ({
  attributes,
  segmentInfo
}) => {
  let segmentAttributes;
  let segmentsFn;
  if (segmentInfo.template) {
    segmentsFn = segmentsFromTemplate;
    segmentAttributes = merge(attributes, segmentInfo.template);
  } else if (segmentInfo.base) {
    segmentsFn = segmentsFromBase;
    segmentAttributes = merge(attributes, segmentInfo.base);
  } else if (segmentInfo.list) {
    segmentsFn = segmentsFromList;
    segmentAttributes = merge(attributes, segmentInfo.list);
  }
  const segmentsInfo = {
    attributes
  };
  if (!segmentsFn) {
    return segmentsInfo;
  }
  const segments = segmentsFn(segmentAttributes, segmentInfo.segmentTimeline);
  if (segmentAttributes.duration) {
    const {
      duration,
      timescale = 1
    } = segmentAttributes;
    segmentAttributes.duration = duration / timescale;
  } else if (segments.length) {
    segmentAttributes.duration = segments.reduce((max, segment) => {
      return Math.max(max, Math.ceil(segment.duration));
    }, 0);
  } else {
    segmentAttributes.duration = 0;
  }
  segmentsInfo.attributes = segmentAttributes;
  segmentsInfo.segments = segments;
  if (segmentInfo.base && segmentAttributes.indexRange) {
    segmentsInfo.sidx = segments[0];
    segmentsInfo.segments = [];
  }
  return segmentsInfo;
};
const toPlaylists = (representations) => representations.map(generateSegments);
const findChildren = (element, name) => from(element.childNodes).filter(({
  tagName
}) => tagName === name);
const getContent = (element) => element.textContent.trim();
const parseDivisionValue = (value) => {
  return parseFloat(value.split("/").reduce((prev, current) => prev / current));
};
const parseDuration = (str) => {
  const SECONDS_IN_YEAR = 365 * 24 * 60 * 60;
  const SECONDS_IN_MONTH = 30 * 24 * 60 * 60;
  const SECONDS_IN_DAY = 24 * 60 * 60;
  const SECONDS_IN_HOUR = 60 * 60;
  const SECONDS_IN_MIN = 60;
  const durationRegex = /P(?:(\d*)Y)?(?:(\d*)M)?(?:(\d*)D)?(?:T(?:(\d*)H)?(?:(\d*)M)?(?:([\d.]*)S)?)?/;
  const match = durationRegex.exec(str);
  if (!match) {
    return 0;
  }
  const [year, month, day, hour, minute, second] = match.slice(1);
  return parseFloat(year || 0) * SECONDS_IN_YEAR + parseFloat(month || 0) * SECONDS_IN_MONTH + parseFloat(day || 0) * SECONDS_IN_DAY + parseFloat(hour || 0) * SECONDS_IN_HOUR + parseFloat(minute || 0) * SECONDS_IN_MIN + parseFloat(second || 0);
};
const parseDate = (str) => {
  const dateRegex = /^\d+-\d+-\d+T\d+:\d+:\d+(\.\d+)?$/;
  if (dateRegex.test(str)) {
    str += "Z";
  }
  return Date.parse(str);
};
const parsers = {
  mediaPresentationDuration(value) {
    return parseDuration(value);
  },
  availabilityStartTime(value) {
    return parseDate(value) / 1e3;
  },
  minimumUpdatePeriod(value) {
    return parseDuration(value);
  },
  suggestedPresentationDelay(value) {
    return parseDuration(value);
  },
  type(value) {
    return value;
  },
  timeShiftBufferDepth(value) {
    return parseDuration(value);
  },
  start(value) {
    return parseDuration(value);
  },
  width(value) {
    return parseInt(value, 10);
  },
  height(value) {
    return parseInt(value, 10);
  },
  bandwidth(value) {
    return parseInt(value, 10);
  },
  frameRate(value) {
    return parseDivisionValue(value);
  },
  startNumber(value) {
    return parseInt(value, 10);
  },
  timescale(value) {
    return parseInt(value, 10);
  },
  presentationTimeOffset(value) {
    return parseInt(value, 10);
  },
  duration(value) {
    const parsedValue = parseInt(value, 10);
    if (isNaN(parsedValue)) {
      return parseDuration(value);
    }
    return parsedValue;
  },
  d(value) {
    return parseInt(value, 10);
  },
  t(value) {
    return parseInt(value, 10);
  },
  r(value) {
    return parseInt(value, 10);
  },
  presentationTime(value) {
    return parseInt(value, 10);
  },
  DEFAULT(value) {
    return value;
  }
};
const parseAttributes = (el) => {
  if (!(el && el.attributes)) {
    return {};
  }
  return from(el.attributes).reduce((a, e) => {
    const parseFn = parsers[e.name] || parsers.DEFAULT;
    a[e.name] = parseFn(e.value);
    return a;
  }, {});
};
const keySystemsMap = {
  "urn:uuid:1077efec-c0b2-4d02-ace3-3c1e52e2fb4b": "org.w3.clearkey",
  "urn:uuid:edef8ba9-79d6-4ace-a3c8-27dcd51d21ed": "com.widevine.alpha",
  "urn:uuid:9a04f079-9840-4286-ab92-e65be0885f95": "com.microsoft.playready",
  "urn:uuid:f239e769-efa3-4850-9c16-a903c6932efb": "com.adobe.primetime",
  "urn:mpeg:dash:mp4protection:2011": "mp4protection"
};
const buildBaseUrls = (references, baseUrlElements) => {
  if (!baseUrlElements.length) {
    return references;
  }
  return flatten(references.map(function(reference) {
    return baseUrlElements.map(function(baseUrlElement) {
      const initialBaseUrl = getContent(baseUrlElement);
      const resolvedBaseUrl = resolveUrl(reference.baseUrl, initialBaseUrl);
      const finalBaseUrl = merge(parseAttributes(baseUrlElement), {
        baseUrl: resolvedBaseUrl
      });
      if (resolvedBaseUrl !== initialBaseUrl && !finalBaseUrl.serviceLocation && reference.serviceLocation) {
        finalBaseUrl.serviceLocation = reference.serviceLocation;
      }
      return finalBaseUrl;
    });
  }));
};
const getSegmentInformation = (adaptationSet) => {
  const segmentTemplate = findChildren(adaptationSet, "SegmentTemplate")[0];
  const segmentList = findChildren(adaptationSet, "SegmentList")[0];
  const segmentUrls = segmentList && findChildren(segmentList, "SegmentURL").map((s) => merge({
    tag: "SegmentURL"
  }, parseAttributes(s)));
  const segmentBase = findChildren(adaptationSet, "SegmentBase")[0];
  const segmentTimelineParentNode = segmentList || segmentTemplate;
  const segmentTimeline = segmentTimelineParentNode && findChildren(segmentTimelineParentNode, "SegmentTimeline")[0];
  const segmentInitializationParentNode = segmentList || segmentBase || segmentTemplate;
  const segmentInitialization = segmentInitializationParentNode && findChildren(segmentInitializationParentNode, "Initialization")[0];
  const template = segmentTemplate && parseAttributes(segmentTemplate);
  if (template && segmentInitialization) {
    template.initialization = segmentInitialization && parseAttributes(segmentInitialization);
  } else if (template && template.initialization) {
    template.initialization = {
      sourceURL: template.initialization
    };
  }
  const segmentInfo = {
    template,
    segmentTimeline: segmentTimeline && findChildren(segmentTimeline, "S").map((s) => parseAttributes(s)),
    list: segmentList && merge(parseAttributes(segmentList), {
      segmentUrls,
      initialization: parseAttributes(segmentInitialization)
    }),
    base: segmentBase && merge(parseAttributes(segmentBase), {
      initialization: parseAttributes(segmentInitialization)
    })
  };
  Object.keys(segmentInfo).forEach((key) => {
    if (!segmentInfo[key]) {
      delete segmentInfo[key];
    }
  });
  return segmentInfo;
};
const inheritBaseUrls = (adaptationSetAttributes, adaptationSetBaseUrls, adaptationSetSegmentInfo) => (representation) => {
  const repBaseUrlElements = findChildren(representation, "BaseURL");
  const repBaseUrls = buildBaseUrls(adaptationSetBaseUrls, repBaseUrlElements);
  const attributes = merge(adaptationSetAttributes, parseAttributes(representation));
  const representationSegmentInfo = getSegmentInformation(representation);
  return repBaseUrls.map((baseUrl) => {
    return {
      segmentInfo: merge(adaptationSetSegmentInfo, representationSegmentInfo),
      attributes: merge(attributes, baseUrl)
    };
  });
};
const generateKeySystemInformation = (contentProtectionNodes) => {
  return contentProtectionNodes.reduce((acc, node) => {
    const attributes = parseAttributes(node);
    if (attributes.schemeIdUri) {
      attributes.schemeIdUri = attributes.schemeIdUri.toLowerCase();
    }
    const keySystem = keySystemsMap[attributes.schemeIdUri];
    if (keySystem) {
      acc[keySystem] = {
        attributes
      };
      const psshNode = findChildren(node, "cenc:pssh")[0];
      if (psshNode) {
        const pssh = getContent(psshNode);
        acc[keySystem].pssh = pssh && decodeB64ToUint8Array(pssh);
      }
    }
    return acc;
  }, {});
};
const parseCaptionServiceMetadata = (service) => {
  if (service.schemeIdUri === "urn:scte:dash:cc:cea-608:2015") {
    const values2 = typeof service.value !== "string" ? [] : service.value.split(";");
    return values2.map((value) => {
      let channel;
      let language;
      language = value;
      if (/^CC\d=/.test(value)) {
        [channel, language] = value.split("=");
      } else if (/^CC\d$/.test(value)) {
        channel = value;
      }
      return {
        channel,
        language
      };
    });
  } else if (service.schemeIdUri === "urn:scte:dash:cc:cea-708:2015") {
    const values2 = typeof service.value !== "string" ? [] : service.value.split(";");
    return values2.map((value) => {
      const flags = {
        "channel": void 0,
        "language": void 0,
        "aspectRatio": 1,
        "easyReader": 0,
        "3D": 0
      };
      if (/=/.test(value)) {
        const [channel, opts = ""] = value.split("=");
        flags.channel = channel;
        flags.language = value;
        opts.split(",").forEach((opt) => {
          const [name, val] = opt.split(":");
          if (name === "lang") {
            flags.language = val;
          } else if (name === "er") {
            flags.easyReader = Number(val);
          } else if (name === "war") {
            flags.aspectRatio = Number(val);
          } else if (name === "3D") {
            flags["3D"] = Number(val);
          }
        });
      } else {
        flags.language = value;
      }
      if (flags.channel) {
        flags.channel = "SERVICE" + flags.channel;
      }
      return flags;
    });
  }
};
const toEventStream = (period) => {
  return flatten(findChildren(period.node, "EventStream").map((eventStream) => {
    const eventStreamAttributes = parseAttributes(eventStream);
    const schemeIdUri = eventStreamAttributes.schemeIdUri;
    return findChildren(eventStream, "Event").map((event) => {
      const eventAttributes = parseAttributes(event);
      const presentationTime = eventAttributes.presentationTime || 0;
      const timescale = eventStreamAttributes.timescale || 1;
      const duration = eventAttributes.duration || 0;
      const start = presentationTime / timescale + period.attributes.start;
      return {
        schemeIdUri,
        value: eventStreamAttributes.value,
        id: eventAttributes.id,
        start,
        end: start + duration / timescale,
        messageData: getContent(event) || eventAttributes.messageData,
        contentEncoding: eventStreamAttributes.contentEncoding,
        presentationTimeOffset: eventStreamAttributes.presentationTimeOffset || 0
      };
    });
  }));
};
const toRepresentations = (periodAttributes, periodBaseUrls, periodSegmentInfo) => (adaptationSet) => {
  const adaptationSetAttributes = parseAttributes(adaptationSet);
  const adaptationSetBaseUrls = buildBaseUrls(periodBaseUrls, findChildren(adaptationSet, "BaseURL"));
  const role = findChildren(adaptationSet, "Role")[0];
  const roleAttributes = {
    role: parseAttributes(role)
  };
  let attrs = merge(periodAttributes, adaptationSetAttributes, roleAttributes);
  const accessibility = findChildren(adaptationSet, "Accessibility")[0];
  const captionServices = parseCaptionServiceMetadata(parseAttributes(accessibility));
  if (captionServices) {
    attrs = merge(attrs, {
      captionServices
    });
  }
  const label = findChildren(adaptationSet, "Label")[0];
  if (label && label.childNodes.length) {
    const labelVal = label.childNodes[0].nodeValue.trim();
    attrs = merge(attrs, {
      label: labelVal
    });
  }
  const contentProtection = generateKeySystemInformation(findChildren(adaptationSet, "ContentProtection"));
  if (Object.keys(contentProtection).length) {
    attrs = merge(attrs, {
      contentProtection
    });
  }
  const segmentInfo = getSegmentInformation(adaptationSet);
  const representations = findChildren(adaptationSet, "Representation");
  const adaptationSetSegmentInfo = merge(periodSegmentInfo, segmentInfo);
  return flatten(representations.map(inheritBaseUrls(attrs, adaptationSetBaseUrls, adaptationSetSegmentInfo)));
};
const toAdaptationSets = (mpdAttributes, mpdBaseUrls) => (period, index) => {
  const periodBaseUrls = buildBaseUrls(mpdBaseUrls, findChildren(period.node, "BaseURL"));
  const periodAttributes = merge(mpdAttributes, {
    periodStart: period.attributes.start
  });
  if (typeof period.attributes.duration === "number") {
    periodAttributes.periodDuration = period.attributes.duration;
  }
  const adaptationSets = findChildren(period.node, "AdaptationSet");
  const periodSegmentInfo = getSegmentInformation(period.node);
  return flatten(adaptationSets.map(toRepresentations(periodAttributes, periodBaseUrls, periodSegmentInfo)));
};
const generateContentSteeringInformation = (contentSteeringNodes, eventHandler) => {
  if (contentSteeringNodes.length > 1) {
    eventHandler({
      type: "warn",
      message: "The MPD manifest should contain no more than one ContentSteering tag"
    });
  }
  if (!contentSteeringNodes.length) {
    return null;
  }
  const infoFromContentSteeringTag = merge({
    serverURL: getContent(contentSteeringNodes[0])
  }, parseAttributes(contentSteeringNodes[0]));
  infoFromContentSteeringTag.queryBeforeStart = infoFromContentSteeringTag.queryBeforeStart === "true";
  return infoFromContentSteeringTag;
};
const getPeriodStart = ({
  attributes,
  priorPeriodAttributes,
  mpdType
}) => {
  if (typeof attributes.start === "number") {
    return attributes.start;
  }
  if (priorPeriodAttributes && typeof priorPeriodAttributes.start === "number" && typeof priorPeriodAttributes.duration === "number") {
    return priorPeriodAttributes.start + priorPeriodAttributes.duration;
  }
  if (!priorPeriodAttributes && mpdType === "static") {
    return 0;
  }
  return null;
};
const inheritAttributes = (mpd, options = {}) => {
  const {
    manifestUri = "",
    NOW = Date.now(),
    clientOffset = 0,
    eventHandler = function() {
    }
  } = options;
  const periodNodes = findChildren(mpd, "Period");
  if (!periodNodes.length) {
    throw new Error(errors.INVALID_NUMBER_OF_PERIOD);
  }
  const locations = findChildren(mpd, "Location");
  const mpdAttributes = parseAttributes(mpd);
  const mpdBaseUrls = buildBaseUrls([{
    baseUrl: manifestUri
  }], findChildren(mpd, "BaseURL"));
  const contentSteeringNodes = findChildren(mpd, "ContentSteering");
  mpdAttributes.type = mpdAttributes.type || "static";
  mpdAttributes.sourceDuration = mpdAttributes.mediaPresentationDuration || 0;
  mpdAttributes.NOW = NOW;
  mpdAttributes.clientOffset = clientOffset;
  if (locations.length) {
    mpdAttributes.locations = locations.map(getContent);
  }
  const periods = [];
  periodNodes.forEach((node, index) => {
    const attributes = parseAttributes(node);
    const priorPeriod = periods[index - 1];
    attributes.start = getPeriodStart({
      attributes,
      priorPeriodAttributes: priorPeriod ? priorPeriod.attributes : null,
      mpdType: mpdAttributes.type
    });
    periods.push({
      node,
      attributes
    });
  });
  return {
    locations: mpdAttributes.locations,
    contentSteeringInfo: generateContentSteeringInformation(contentSteeringNodes, eventHandler),
    representationInfo: flatten(periods.map(toAdaptationSets(mpdAttributes, mpdBaseUrls))),
    eventStream: flatten(periods.map(toEventStream))
  };
};
const stringToMpdXml = (manifestString) => {
  if (manifestString === "") {
    throw new Error(errors.DASH_EMPTY_MANIFEST);
  }
  const parser = new DOMParser();
  let xml;
  let mpd;
  try {
    xml = parser.parseFromString(manifestString, "application/xml");
    mpd = xml && xml.documentElement.tagName === "MPD" ? xml.documentElement : null;
  } catch (e) {
  }
  if (!mpd || mpd && mpd.getElementsByTagName("parsererror").length > 0) {
    throw new Error(errors.DASH_INVALID_XML);
  }
  return mpd;
};
const parseUTCTimingScheme = (mpd) => {
  const UTCTimingNode = findChildren(mpd, "UTCTiming")[0];
  if (!UTCTimingNode) {
    return null;
  }
  const attributes = parseAttributes(UTCTimingNode);
  switch (attributes.schemeIdUri) {
    case "urn:mpeg:dash:utc:http-head:2014":
    case "urn:mpeg:dash:utc:http-head:2012":
      attributes.method = "HEAD";
      break;
    case "urn:mpeg:dash:utc:http-xsdate:2014":
    case "urn:mpeg:dash:utc:http-iso:2014":
    case "urn:mpeg:dash:utc:http-xsdate:2012":
    case "urn:mpeg:dash:utc:http-iso:2012":
      attributes.method = "GET";
      break;
    case "urn:mpeg:dash:utc:direct:2014":
    case "urn:mpeg:dash:utc:direct:2012":
      attributes.method = "DIRECT";
      attributes.value = Date.parse(attributes.value);
      break;
    case "urn:mpeg:dash:utc:http-ntp:2014":
    case "urn:mpeg:dash:utc:ntp:2014":
    case "urn:mpeg:dash:utc:sntp:2014":
    default:
      throw new Error(errors.UNSUPPORTED_UTC_TIMING_SCHEME);
  }
  return attributes;
};
const parse = (manifestString, options = {}) => {
  const parsedManifestInfo = inheritAttributes(stringToMpdXml(manifestString), options);
  const playlists = toPlaylists(parsedManifestInfo.representationInfo);
  return toM3u8({
    dashPlaylists: playlists,
    locations: parsedManifestInfo.locations,
    contentSteering: parsedManifestInfo.contentSteeringInfo,
    sidxMapping: options.sidxMapping,
    previousManifest: options.previousManifest,
    eventStream: parsedManifestInfo.eventStream
  });
};
const parseUTCTiming = (manifestString) => parseUTCTimingScheme(stringToMpdXml(manifestString));
export {
  parse as a,
  addSidxSegmentsToPlaylist$1 as b,
  generateSidxKey as g,
  parseUTCTiming as p
};
