import { d as document_1, w as window_1 } from "./global.dfc622c4.js";
var browserIndex = { exports: {} };
var document = document_1;
var _objCreate = Object.create || function() {
  function F() {
  }
  return function(o) {
    if (arguments.length !== 1) {
      throw new Error("Object.create shim only accepts one parameter.");
    }
    F.prototype = o;
    return new F();
  };
}();
function ParsingError(errorData, message) {
  this.name = "ParsingError";
  this.code = errorData.code;
  this.message = message || errorData.message;
}
ParsingError.prototype = _objCreate(Error.prototype);
ParsingError.prototype.constructor = ParsingError;
ParsingError.Errors = {
  BadSignature: {
    code: 0,
    message: "Malformed WebVTT signature."
  },
  BadTimeStamp: {
    code: 1,
    message: "Malformed time stamp."
  }
};
function parseTimeStamp(input) {
  function computeSeconds(h, m2, s, f) {
    return (h | 0) * 3600 + (m2 | 0) * 60 + (s | 0) + (f | 0) / 1e3;
  }
  var m = input.match(/^(\d+):(\d{1,2})(:\d{1,2})?\.(\d{3})/);
  if (!m) {
    return null;
  }
  if (m[3]) {
    return computeSeconds(m[1], m[2], m[3].replace(":", ""), m[4]);
  } else if (m[1] > 59) {
    return computeSeconds(m[1], m[2], 0, m[4]);
  } else {
    return computeSeconds(0, m[1], m[2], m[4]);
  }
}
function Settings() {
  this.values = _objCreate(null);
}
Settings.prototype = {
  set: function(k, v) {
    if (!this.get(k) && v !== "") {
      this.values[k] = v;
    }
  },
  get: function(k, dflt, defaultKey) {
    if (defaultKey) {
      return this.has(k) ? this.values[k] : dflt[defaultKey];
    }
    return this.has(k) ? this.values[k] : dflt;
  },
  has: function(k) {
    return k in this.values;
  },
  alt: function(k, v, a) {
    for (var n = 0; n < a.length; ++n) {
      if (v === a[n]) {
        this.set(k, v);
        break;
      }
    }
  },
  integer: function(k, v) {
    if (/^-?\d+$/.test(v)) {
      this.set(k, parseInt(v, 10));
    }
  },
  percent: function(k, v) {
    if (v.match(/^([\d]{1,3})(\.[\d]*)?%$/)) {
      v = parseFloat(v);
      if (v >= 0 && v <= 100) {
        this.set(k, v);
        return true;
      }
    }
    return false;
  }
};
function parseOptions(input, callback, keyValueDelim, groupDelim) {
  var groups = groupDelim ? input.split(groupDelim) : [input];
  for (var i in groups) {
    if (typeof groups[i] !== "string") {
      continue;
    }
    var kv = groups[i].split(keyValueDelim);
    if (kv.length !== 2) {
      continue;
    }
    var k = kv[0].trim();
    var v = kv[1].trim();
    callback(k, v);
  }
}
function parseCue(input, cue, regionList) {
  var oInput = input;
  function consumeTimeStamp() {
    var ts = parseTimeStamp(input);
    if (ts === null) {
      throw new ParsingError(
        ParsingError.Errors.BadTimeStamp,
        "Malformed timestamp: " + oInput
      );
    }
    input = input.replace(/^[^\sa-zA-Z-]+/, "");
    return ts;
  }
  function consumeCueSettings(input2, cue2) {
    var settings = new Settings();
    parseOptions(input2, function(k, v) {
      switch (k) {
        case "region":
          for (var i = regionList.length - 1; i >= 0; i--) {
            if (regionList[i].id === v) {
              settings.set(k, regionList[i].region);
              break;
            }
          }
          break;
        case "vertical":
          settings.alt(k, v, ["rl", "lr"]);
          break;
        case "line":
          var vals = v.split(","), vals0 = vals[0];
          settings.integer(k, vals0);
          settings.percent(k, vals0) ? settings.set("snapToLines", false) : null;
          settings.alt(k, vals0, ["auto"]);
          if (vals.length === 2) {
            settings.alt("lineAlign", vals[1], ["start", "center", "end"]);
          }
          break;
        case "position":
          vals = v.split(",");
          settings.percent(k, vals[0]);
          if (vals.length === 2) {
            settings.alt("positionAlign", vals[1], ["start", "center", "end"]);
          }
          break;
        case "size":
          settings.percent(k, v);
          break;
        case "align":
          settings.alt(k, v, ["start", "center", "end", "left", "right"]);
          break;
      }
    }, /:/, /\s/);
    cue2.region = settings.get("region", null);
    cue2.vertical = settings.get("vertical", "");
    try {
      cue2.line = settings.get("line", "auto");
    } catch (e) {
    }
    cue2.lineAlign = settings.get("lineAlign", "start");
    cue2.snapToLines = settings.get("snapToLines", true);
    cue2.size = settings.get("size", 100);
    try {
      cue2.align = settings.get("align", "center");
    } catch (e) {
      cue2.align = settings.get("align", "middle");
    }
    try {
      cue2.position = settings.get("position", "auto");
    } catch (e) {
      cue2.position = settings.get("position", {
        start: 0,
        left: 0,
        center: 50,
        middle: 50,
        end: 100,
        right: 100
      }, cue2.align);
    }
    cue2.positionAlign = settings.get("positionAlign", {
      start: "start",
      left: "start",
      center: "center",
      middle: "center",
      end: "end",
      right: "end"
    }, cue2.align);
  }
  function skipWhitespace() {
    input = input.replace(/^\s+/, "");
  }
  skipWhitespace();
  cue.startTime = consumeTimeStamp();
  skipWhitespace();
  if (input.substr(0, 3) !== "-->") {
    throw new ParsingError(
      ParsingError.Errors.BadTimeStamp,
      "Malformed time stamp (time stamps must be separated by '-->'): " + oInput
    );
  }
  input = input.substr(3);
  skipWhitespace();
  cue.endTime = consumeTimeStamp();
  skipWhitespace();
  consumeCueSettings(input, cue);
}
var TEXTAREA_ELEMENT = document.createElement && document.createElement("textarea");
var TAG_NAME = {
  c: "span",
  i: "i",
  b: "b",
  u: "u",
  ruby: "ruby",
  rt: "rt",
  v: "span",
  lang: "span"
};
var DEFAULT_COLOR_CLASS = {
  white: "rgba(255,255,255,1)",
  lime: "rgba(0,255,0,1)",
  cyan: "rgba(0,255,255,1)",
  red: "rgba(255,0,0,1)",
  yellow: "rgba(255,255,0,1)",
  magenta: "rgba(255,0,255,1)",
  blue: "rgba(0,0,255,1)",
  black: "rgba(0,0,0,1)"
};
var TAG_ANNOTATION = {
  v: "title",
  lang: "lang"
};
var NEEDS_PARENT = {
  rt: "ruby"
};
function parseContent(window2, input) {
  function nextToken() {
    if (!input) {
      return null;
    }
    function consume(result) {
      input = input.substr(result.length);
      return result;
    }
    var m2 = input.match(/^([^<]*)(<[^>]*>?)?/);
    return consume(m2[1] ? m2[1] : m2[2]);
  }
  function unescape(s) {
    TEXTAREA_ELEMENT.innerHTML = s;
    s = TEXTAREA_ELEMENT.textContent;
    TEXTAREA_ELEMENT.textContent = "";
    return s;
  }
  function shouldAdd(current2, element) {
    return !NEEDS_PARENT[element.localName] || NEEDS_PARENT[element.localName] === current2.localName;
  }
  function createElement(type, annotation) {
    var tagName = TAG_NAME[type];
    if (!tagName) {
      return null;
    }
    var element = window2.document.createElement(tagName);
    var name = TAG_ANNOTATION[type];
    if (name && annotation) {
      element[name] = annotation.trim();
    }
    return element;
  }
  var rootDiv = window2.document.createElement("div"), current = rootDiv, t, tagStack = [];
  while ((t = nextToken()) !== null) {
    if (t[0] === "<") {
      if (t[1] === "/") {
        if (tagStack.length && tagStack[tagStack.length - 1] === t.substr(2).replace(">", "")) {
          tagStack.pop();
          current = current.parentNode;
        }
        continue;
      }
      var ts = parseTimeStamp(t.substr(1, t.length - 2));
      var node;
      if (ts) {
        node = window2.document.createProcessingInstruction("timestamp", ts);
        current.appendChild(node);
        continue;
      }
      var m = t.match(/^<([^.\s/0-9>]+)(\.[^\s\\>]+)?([^>\\]+)?(\\?)>?$/);
      if (!m) {
        continue;
      }
      node = createElement(m[1], m[3]);
      if (!node) {
        continue;
      }
      if (!shouldAdd(current, node)) {
        continue;
      }
      if (m[2]) {
        var classes = m[2].split(".");
        classes.forEach(function(cl) {
          var bgColor = /^bg_/.test(cl);
          var colorName = bgColor ? cl.slice(3) : cl;
          if (DEFAULT_COLOR_CLASS.hasOwnProperty(colorName)) {
            var propName = bgColor ? "background-color" : "color";
            var propValue = DEFAULT_COLOR_CLASS[colorName];
            node.style[propName] = propValue;
          }
        });
        node.className = classes.join(" ");
      }
      tagStack.push(m[1]);
      current.appendChild(node);
      current = node;
      continue;
    }
    current.appendChild(window2.document.createTextNode(unescape(t)));
  }
  return rootDiv;
}
var strongRTLRanges = [
  [1470, 1470],
  [1472, 1472],
  [1475, 1475],
  [1478, 1478],
  [1488, 1514],
  [1520, 1524],
  [1544, 1544],
  [1547, 1547],
  [1549, 1549],
  [1563, 1563],
  [1566, 1610],
  [1645, 1647],
  [1649, 1749],
  [1765, 1766],
  [1774, 1775],
  [1786, 1805],
  [1807, 1808],
  [1810, 1839],
  [1869, 1957],
  [1969, 1969],
  [1984, 2026],
  [2036, 2037],
  [2042, 2042],
  [2048, 2069],
  [2074, 2074],
  [2084, 2084],
  [2088, 2088],
  [2096, 2110],
  [2112, 2136],
  [2142, 2142],
  [2208, 2208],
  [2210, 2220],
  [8207, 8207],
  [64285, 64285],
  [64287, 64296],
  [64298, 64310],
  [64312, 64316],
  [64318, 64318],
  [64320, 64321],
  [64323, 64324],
  [64326, 64449],
  [64467, 64829],
  [64848, 64911],
  [64914, 64967],
  [65008, 65020],
  [65136, 65140],
  [65142, 65276],
  [67584, 67589],
  [67592, 67592],
  [67594, 67637],
  [67639, 67640],
  [67644, 67644],
  [67647, 67669],
  [67671, 67679],
  [67840, 67867],
  [67872, 67897],
  [67903, 67903],
  [67968, 68023],
  [68030, 68031],
  [68096, 68096],
  [68112, 68115],
  [68117, 68119],
  [68121, 68147],
  [68160, 68167],
  [68176, 68184],
  [68192, 68223],
  [68352, 68405],
  [68416, 68437],
  [68440, 68466],
  [68472, 68479],
  [68608, 68680],
  [126464, 126467],
  [126469, 126495],
  [126497, 126498],
  [126500, 126500],
  [126503, 126503],
  [126505, 126514],
  [126516, 126519],
  [126521, 126521],
  [126523, 126523],
  [126530, 126530],
  [126535, 126535],
  [126537, 126537],
  [126539, 126539],
  [126541, 126543],
  [126545, 126546],
  [126548, 126548],
  [126551, 126551],
  [126553, 126553],
  [126555, 126555],
  [126557, 126557],
  [126559, 126559],
  [126561, 126562],
  [126564, 126564],
  [126567, 126570],
  [126572, 126578],
  [126580, 126583],
  [126585, 126588],
  [126590, 126590],
  [126592, 126601],
  [126603, 126619],
  [126625, 126627],
  [126629, 126633],
  [126635, 126651],
  [1114109, 1114109]
];
function isStrongRTLChar(charCode) {
  for (var i = 0; i < strongRTLRanges.length; i++) {
    var currentRange = strongRTLRanges[i];
    if (charCode >= currentRange[0] && charCode <= currentRange[1]) {
      return true;
    }
  }
  return false;
}
function determineBidi(cueDiv) {
  var nodeStack = [], text = "", charCode;
  if (!cueDiv || !cueDiv.childNodes) {
    return "ltr";
  }
  function pushNodes(nodeStack2, node) {
    for (var i2 = node.childNodes.length - 1; i2 >= 0; i2--) {
      nodeStack2.push(node.childNodes[i2]);
    }
  }
  function nextTextNode(nodeStack2) {
    if (!nodeStack2 || !nodeStack2.length) {
      return null;
    }
    var node = nodeStack2.pop(), text2 = node.textContent || node.innerText;
    if (text2) {
      var m = text2.match(/^.*(\n|\r)/);
      if (m) {
        nodeStack2.length = 0;
        return m[0];
      }
      return text2;
    }
    if (node.tagName === "ruby") {
      return nextTextNode(nodeStack2);
    }
    if (node.childNodes) {
      pushNodes(nodeStack2, node);
      return nextTextNode(nodeStack2);
    }
  }
  pushNodes(nodeStack, cueDiv);
  while (text = nextTextNode(nodeStack)) {
    for (var i = 0; i < text.length; i++) {
      charCode = text.charCodeAt(i);
      if (isStrongRTLChar(charCode)) {
        return "rtl";
      }
    }
  }
  return "ltr";
}
function computeLinePos(cue) {
  if (typeof cue.line === "number" && (cue.snapToLines || cue.line >= 0 && cue.line <= 100)) {
    return cue.line;
  }
  if (!cue.track || !cue.track.textTrackList || !cue.track.textTrackList.mediaElement) {
    return -1;
  }
  var track = cue.track, trackList = track.textTrackList, count = 0;
  for (var i = 0; i < trackList.length && trackList[i] !== track; i++) {
    if (trackList[i].mode === "showing") {
      count++;
    }
  }
  return ++count * -1;
}
function StyleBox() {
}
StyleBox.prototype.applyStyles = function(styles, div) {
  div = div || this.div;
  for (var prop in styles) {
    if (styles.hasOwnProperty(prop)) {
      div.style[prop] = styles[prop];
    }
  }
};
StyleBox.prototype.formatStyle = function(val, unit) {
  return val === 0 ? 0 : val + unit;
};
function CueStyleBox(window2, cue, styleOptions) {
  StyleBox.call(this);
  this.cue = cue;
  this.cueDiv = parseContent(window2, cue.text);
  var styles = {
    color: "rgba(255, 255, 255, 1)",
    backgroundColor: "rgba(0, 0, 0, 0.8)",
    position: "relative",
    left: 0,
    right: 0,
    top: 0,
    bottom: 0,
    display: "inline",
    writingMode: cue.vertical === "" ? "horizontal-tb" : cue.vertical === "lr" ? "vertical-lr" : "vertical-rl",
    unicodeBidi: "plaintext"
  };
  this.applyStyles(styles, this.cueDiv);
  this.div = window2.document.createElement("div");
  styles = {
    direction: determineBidi(this.cueDiv),
    writingMode: cue.vertical === "" ? "horizontal-tb" : cue.vertical === "lr" ? "vertical-lr" : "vertical-rl",
    unicodeBidi: "plaintext",
    textAlign: cue.align === "middle" ? "center" : cue.align,
    font: styleOptions.font,
    whiteSpace: "pre-line",
    position: "absolute"
  };
  this.applyStyles(styles);
  this.div.appendChild(this.cueDiv);
  var textPos = 0;
  switch (cue.positionAlign) {
    case "start":
    case "line-left":
      textPos = cue.position;
      break;
    case "center":
      textPos = cue.position - cue.size / 2;
      break;
    case "end":
    case "line-right":
      textPos = cue.position - cue.size;
      break;
  }
  if (cue.vertical === "") {
    this.applyStyles({
      left: this.formatStyle(textPos, "%"),
      width: this.formatStyle(cue.size, "%")
    });
  } else {
    this.applyStyles({
      top: this.formatStyle(textPos, "%"),
      height: this.formatStyle(cue.size, "%")
    });
  }
  this.move = function(box) {
    this.applyStyles({
      top: this.formatStyle(box.top, "px"),
      bottom: this.formatStyle(box.bottom, "px"),
      left: this.formatStyle(box.left, "px"),
      right: this.formatStyle(box.right, "px"),
      height: this.formatStyle(box.height, "px"),
      width: this.formatStyle(box.width, "px")
    });
  };
}
CueStyleBox.prototype = _objCreate(StyleBox.prototype);
CueStyleBox.prototype.constructor = CueStyleBox;
function BoxPosition(obj) {
  var lh, height, width, top;
  if (obj.div) {
    height = obj.div.offsetHeight;
    width = obj.div.offsetWidth;
    top = obj.div.offsetTop;
    var rects = (rects = obj.div.childNodes) && (rects = rects[0]) && rects.getClientRects && rects.getClientRects();
    obj = obj.div.getBoundingClientRect();
    lh = rects ? Math.max(rects[0] && rects[0].height || 0, obj.height / rects.length) : 0;
  }
  this.left = obj.left;
  this.right = obj.right;
  this.top = obj.top || top;
  this.height = obj.height || height;
  this.bottom = obj.bottom || top + (obj.height || height);
  this.width = obj.width || width;
  this.lineHeight = lh !== void 0 ? lh : obj.lineHeight;
}
BoxPosition.prototype.move = function(axis, toMove) {
  toMove = toMove !== void 0 ? toMove : this.lineHeight;
  switch (axis) {
    case "+x":
      this.left += toMove;
      this.right += toMove;
      break;
    case "-x":
      this.left -= toMove;
      this.right -= toMove;
      break;
    case "+y":
      this.top += toMove;
      this.bottom += toMove;
      break;
    case "-y":
      this.top -= toMove;
      this.bottom -= toMove;
      break;
  }
};
BoxPosition.prototype.overlaps = function(b2) {
  return this.left < b2.right && this.right > b2.left && this.top < b2.bottom && this.bottom > b2.top;
};
BoxPosition.prototype.overlapsAny = function(boxes) {
  for (var i = 0; i < boxes.length; i++) {
    if (this.overlaps(boxes[i])) {
      return true;
    }
  }
  return false;
};
BoxPosition.prototype.within = function(container) {
  return this.top >= container.top && this.bottom <= container.bottom && this.left >= container.left && this.right <= container.right;
};
BoxPosition.prototype.overlapsOppositeAxis = function(container, axis) {
  switch (axis) {
    case "+x":
      return this.left < container.left;
    case "-x":
      return this.right > container.right;
    case "+y":
      return this.top < container.top;
    case "-y":
      return this.bottom > container.bottom;
  }
};
BoxPosition.prototype.intersectPercentage = function(b2) {
  var x = Math.max(0, Math.min(this.right, b2.right) - Math.max(this.left, b2.left)), y = Math.max(0, Math.min(this.bottom, b2.bottom) - Math.max(this.top, b2.top)), intersectArea = x * y;
  return intersectArea / (this.height * this.width);
};
BoxPosition.prototype.toCSSCompatValues = function(reference) {
  return {
    top: this.top - reference.top,
    bottom: reference.bottom - this.bottom,
    left: this.left - reference.left,
    right: reference.right - this.right,
    height: this.height,
    width: this.width
  };
};
BoxPosition.getSimpleBoxPosition = function(obj) {
  var height = obj.div ? obj.div.offsetHeight : obj.tagName ? obj.offsetHeight : 0;
  var width = obj.div ? obj.div.offsetWidth : obj.tagName ? obj.offsetWidth : 0;
  var top = obj.div ? obj.div.offsetTop : obj.tagName ? obj.offsetTop : 0;
  obj = obj.div ? obj.div.getBoundingClientRect() : obj.tagName ? obj.getBoundingClientRect() : obj;
  var ret = {
    left: obj.left,
    right: obj.right,
    top: obj.top || top,
    height: obj.height || height,
    bottom: obj.bottom || top + (obj.height || height),
    width: obj.width || width
  };
  return ret;
};
function moveBoxToLinePosition(window2, styleBox, containerBox, boxPositions) {
  function findBestPosition(b, axis2) {
    var bestPosition2, specifiedPosition = new BoxPosition(b), percentage = 1;
    for (var i = 0; i < axis2.length; i++) {
      while (b.overlapsOppositeAxis(containerBox, axis2[i]) || b.within(containerBox) && b.overlapsAny(boxPositions)) {
        b.move(axis2[i]);
      }
      if (b.within(containerBox)) {
        return b;
      }
      var p = b.intersectPercentage(containerBox);
      if (percentage > p) {
        bestPosition2 = new BoxPosition(b);
        percentage = p;
      }
      b = new BoxPosition(specifiedPosition);
    }
    return bestPosition2 || specifiedPosition;
  }
  var boxPosition = new BoxPosition(styleBox), cue = styleBox.cue, linePos = computeLinePos(cue), axis = [];
  if (cue.snapToLines) {
    var size;
    switch (cue.vertical) {
      case "":
        axis = ["+y", "-y"];
        size = "height";
        break;
      case "rl":
        axis = ["+x", "-x"];
        size = "width";
        break;
      case "lr":
        axis = ["-x", "+x"];
        size = "width";
        break;
    }
    var step = boxPosition.lineHeight, position = step * Math.round(linePos), maxPosition = containerBox[size] + step, initialAxis = axis[0];
    if (Math.abs(position) > maxPosition) {
      position = position < 0 ? -1 : 1;
      position *= Math.ceil(maxPosition / step) * step;
    }
    if (linePos < 0) {
      position += cue.vertical === "" ? containerBox.height : containerBox.width;
      axis = axis.reverse();
    }
    boxPosition.move(initialAxis, position);
  } else {
    var calculatedPercentage = boxPosition.lineHeight / containerBox.height * 100;
    switch (cue.lineAlign) {
      case "center":
        linePos -= calculatedPercentage / 2;
        break;
      case "end":
        linePos -= calculatedPercentage;
        break;
    }
    switch (cue.vertical) {
      case "":
        styleBox.applyStyles({
          top: styleBox.formatStyle(linePos, "%")
        });
        break;
      case "rl":
        styleBox.applyStyles({
          left: styleBox.formatStyle(linePos, "%")
        });
        break;
      case "lr":
        styleBox.applyStyles({
          right: styleBox.formatStyle(linePos, "%")
        });
        break;
    }
    axis = ["+y", "-x", "+x", "-y"];
    boxPosition = new BoxPosition(styleBox);
  }
  var bestPosition = findBestPosition(boxPosition, axis);
  styleBox.move(bestPosition.toCSSCompatValues(containerBox));
}
function WebVTT$1() {
}
WebVTT$1.StringDecoder = function() {
  return {
    decode: function(data) {
      if (!data) {
        return "";
      }
      if (typeof data !== "string") {
        throw new Error("Error - expected string data.");
      }
      return decodeURIComponent(encodeURIComponent(data));
    }
  };
};
WebVTT$1.convertCueToDOMTree = function(window2, cuetext) {
  if (!window2 || !cuetext) {
    return null;
  }
  return parseContent(window2, cuetext);
};
var FONT_SIZE_PERCENT = 0.05;
var FONT_STYLE = "sans-serif";
var CUE_BACKGROUND_PADDING = "1.5%";
WebVTT$1.processCues = function(window2, cues, overlay) {
  if (!window2 || !cues || !overlay) {
    return null;
  }
  while (overlay.firstChild) {
    overlay.removeChild(overlay.firstChild);
  }
  var paddedOverlay = window2.document.createElement("div");
  paddedOverlay.style.position = "absolute";
  paddedOverlay.style.left = "0";
  paddedOverlay.style.right = "0";
  paddedOverlay.style.top = "0";
  paddedOverlay.style.bottom = "0";
  paddedOverlay.style.margin = CUE_BACKGROUND_PADDING;
  overlay.appendChild(paddedOverlay);
  function shouldCompute(cues2) {
    for (var i2 = 0; i2 < cues2.length; i2++) {
      if (cues2[i2].hasBeenReset || !cues2[i2].displayState) {
        return true;
      }
    }
    return false;
  }
  if (!shouldCompute(cues)) {
    for (var i = 0; i < cues.length; i++) {
      paddedOverlay.appendChild(cues[i].displayState);
    }
    return;
  }
  var boxPositions = [], containerBox = BoxPosition.getSimpleBoxPosition(paddedOverlay), fontSize = Math.round(containerBox.height * FONT_SIZE_PERCENT * 100) / 100;
  var styleOptions = {
    font: fontSize + "px " + FONT_STYLE
  };
  (function() {
    var styleBox, cue;
    for (var i2 = 0; i2 < cues.length; i2++) {
      cue = cues[i2];
      styleBox = new CueStyleBox(window2, cue, styleOptions);
      paddedOverlay.appendChild(styleBox.div);
      moveBoxToLinePosition(window2, styleBox, containerBox, boxPositions);
      cue.displayState = styleBox.div;
      boxPositions.push(BoxPosition.getSimpleBoxPosition(styleBox));
    }
  })();
};
WebVTT$1.Parser = function(window2, vttjs2, decoder) {
  if (!decoder) {
    decoder = vttjs2;
    vttjs2 = {};
  }
  if (!vttjs2) {
    vttjs2 = {};
  }
  this.window = window2;
  this.vttjs = vttjs2;
  this.state = "INITIAL";
  this.buffer = "";
  this.decoder = decoder || new TextDecoder("utf8");
  this.regionList = [];
};
WebVTT$1.Parser.prototype = {
  reportOrThrowError: function(e) {
    if (e instanceof ParsingError) {
      this.onparsingerror && this.onparsingerror(e);
    } else {
      throw e;
    }
  },
  parse: function(data) {
    var self = this;
    if (data) {
      self.buffer += self.decoder.decode(data, { stream: true });
    }
    function collectNextLine() {
      var buffer = self.buffer;
      var pos = 0;
      while (pos < buffer.length && buffer[pos] !== "\r" && buffer[pos] !== "\n") {
        ++pos;
      }
      var line2 = buffer.substr(0, pos);
      if (buffer[pos] === "\r") {
        ++pos;
      }
      if (buffer[pos] === "\n") {
        ++pos;
      }
      self.buffer = buffer.substr(pos);
      return line2;
    }
    function parseRegion(input) {
      var settings = new Settings();
      parseOptions(input, function(k, v) {
        switch (k) {
          case "id":
            settings.set(k, v);
            break;
          case "width":
            settings.percent(k, v);
            break;
          case "lines":
            settings.integer(k, v);
            break;
          case "regionanchor":
          case "viewportanchor":
            var xy = v.split(",");
            if (xy.length !== 2) {
              break;
            }
            var anchor = new Settings();
            anchor.percent("x", xy[0]);
            anchor.percent("y", xy[1]);
            if (!anchor.has("x") || !anchor.has("y")) {
              break;
            }
            settings.set(k + "X", anchor.get("x"));
            settings.set(k + "Y", anchor.get("y"));
            break;
          case "scroll":
            settings.alt(k, v, ["up"]);
            break;
        }
      }, /=/, /\s/);
      if (settings.has("id")) {
        var region = new (self.vttjs.VTTRegion || self.window.VTTRegion)();
        region.width = settings.get("width", 100);
        region.lines = settings.get("lines", 3);
        region.regionAnchorX = settings.get("regionanchorX", 0);
        region.regionAnchorY = settings.get("regionanchorY", 100);
        region.viewportAnchorX = settings.get("viewportanchorX", 0);
        region.viewportAnchorY = settings.get("viewportanchorY", 100);
        region.scroll = settings.get("scroll", "");
        self.onregion && self.onregion(region);
        self.regionList.push({
          id: settings.get("id"),
          region
        });
      }
    }
    function parseTimestampMap(input) {
      var settings = new Settings();
      parseOptions(input, function(k, v) {
        switch (k) {
          case "MPEGT":
            settings.integer(k + "S", v);
            break;
          case "LOCA":
            settings.set(k + "L", parseTimeStamp(v));
            break;
        }
      }, /[^\d]:/, /,/);
      self.ontimestampmap && self.ontimestampmap({
        "MPEGTS": settings.get("MPEGTS"),
        "LOCAL": settings.get("LOCAL")
      });
    }
    function parseHeader(input) {
      if (input.match(/X-TIMESTAMP-MAP/)) {
        parseOptions(input, function(k, v) {
          switch (k) {
            case "X-TIMESTAMP-MAP":
              parseTimestampMap(v);
              break;
          }
        }, /=/);
      } else {
        parseOptions(input, function(k, v) {
          switch (k) {
            case "Region":
              parseRegion(v);
              break;
          }
        }, /:/);
      }
    }
    try {
      var line;
      if (self.state === "INITIAL") {
        if (!/\r\n|\n/.test(self.buffer)) {
          return this;
        }
        line = collectNextLine();
        var m = line.match(/^WEBVTT([ \t].*)?$/);
        if (!m || !m[0]) {
          throw new ParsingError(ParsingError.Errors.BadSignature);
        }
        self.state = "HEADER";
      }
      var alreadyCollectedLine = false;
      while (self.buffer) {
        if (!/\r\n|\n/.test(self.buffer)) {
          return this;
        }
        if (!alreadyCollectedLine) {
          line = collectNextLine();
        } else {
          alreadyCollectedLine = false;
        }
        switch (self.state) {
          case "HEADER":
            if (/:/.test(line)) {
              parseHeader(line);
            } else if (!line) {
              self.state = "ID";
            }
            continue;
          case "NOTE":
            if (!line) {
              self.state = "ID";
            }
            continue;
          case "ID":
            if (/^NOTE($|[ \t])/.test(line)) {
              self.state = "NOTE";
              break;
            }
            if (!line) {
              continue;
            }
            self.cue = new (self.vttjs.VTTCue || self.window.VTTCue)(0, 0, "");
            try {
              self.cue.align = "center";
            } catch (e) {
              self.cue.align = "middle";
            }
            self.state = "CUE";
            if (line.indexOf("-->") === -1) {
              self.cue.id = line;
              continue;
            }
          case "CUE":
            try {
              parseCue(line, self.cue, self.regionList);
            } catch (e) {
              self.reportOrThrowError(e);
              self.cue = null;
              self.state = "BADCUE";
              continue;
            }
            self.state = "CUETEXT";
            continue;
          case "CUETEXT":
            var hasSubstring = line.indexOf("-->") !== -1;
            if (!line || hasSubstring && (alreadyCollectedLine = true)) {
              self.oncue && self.oncue(self.cue);
              self.cue = null;
              self.state = "ID";
              continue;
            }
            if (self.cue.text) {
              self.cue.text += "\n";
            }
            self.cue.text += line.replace(/\u2028/g, "\n").replace(/u2029/g, "\n");
            continue;
          case "BADCUE":
            if (!line) {
              self.state = "ID";
            }
            continue;
        }
      }
    } catch (e) {
      self.reportOrThrowError(e);
      if (self.state === "CUETEXT" && self.cue && self.oncue) {
        self.oncue(self.cue);
      }
      self.cue = null;
      self.state = self.state === "INITIAL" ? "BADWEBVTT" : "BADCUE";
    }
    return this;
  },
  flush: function() {
    var self = this;
    try {
      self.buffer += self.decoder.decode();
      if (self.cue || self.state === "HEADER") {
        self.buffer += "\n\n";
        self.parse();
      }
      if (self.state === "INITIAL") {
        throw new ParsingError(ParsingError.Errors.BadSignature);
      }
    } catch (e) {
      self.reportOrThrowError(e);
    }
    self.onflush && self.onflush();
    return this;
  }
};
var vtt = WebVTT$1;
var autoKeyword = "auto";
var directionSetting = {
  "": 1,
  "lr": 1,
  "rl": 1
};
var alignSetting = {
  "start": 1,
  "center": 1,
  "end": 1,
  "left": 1,
  "right": 1,
  "auto": 1,
  "line-left": 1,
  "line-right": 1
};
function findDirectionSetting(value) {
  if (typeof value !== "string") {
    return false;
  }
  var dir = directionSetting[value.toLowerCase()];
  return dir ? value.toLowerCase() : false;
}
function findAlignSetting(value) {
  if (typeof value !== "string") {
    return false;
  }
  var align = alignSetting[value.toLowerCase()];
  return align ? value.toLowerCase() : false;
}
function VTTCue(startTime, endTime, text) {
  this.hasBeenReset = false;
  var _id = "";
  var _pauseOnExit = false;
  var _startTime = startTime;
  var _endTime = endTime;
  var _text = text;
  var _region = null;
  var _vertical = "";
  var _snapToLines = true;
  var _line = "auto";
  var _lineAlign = "start";
  var _position = "auto";
  var _positionAlign = "auto";
  var _size = 100;
  var _align = "center";
  Object.defineProperties(this, {
    "id": {
      enumerable: true,
      get: function() {
        return _id;
      },
      set: function(value) {
        _id = "" + value;
      }
    },
    "pauseOnExit": {
      enumerable: true,
      get: function() {
        return _pauseOnExit;
      },
      set: function(value) {
        _pauseOnExit = !!value;
      }
    },
    "startTime": {
      enumerable: true,
      get: function() {
        return _startTime;
      },
      set: function(value) {
        if (typeof value !== "number") {
          throw new TypeError("Start time must be set to a number.");
        }
        _startTime = value;
        this.hasBeenReset = true;
      }
    },
    "endTime": {
      enumerable: true,
      get: function() {
        return _endTime;
      },
      set: function(value) {
        if (typeof value !== "number") {
          throw new TypeError("End time must be set to a number.");
        }
        _endTime = value;
        this.hasBeenReset = true;
      }
    },
    "text": {
      enumerable: true,
      get: function() {
        return _text;
      },
      set: function(value) {
        _text = "" + value;
        this.hasBeenReset = true;
      }
    },
    "region": {
      enumerable: true,
      get: function() {
        return _region;
      },
      set: function(value) {
        _region = value;
        this.hasBeenReset = true;
      }
    },
    "vertical": {
      enumerable: true,
      get: function() {
        return _vertical;
      },
      set: function(value) {
        var setting = findDirectionSetting(value);
        if (setting === false) {
          throw new SyntaxError("Vertical: an invalid or illegal direction string was specified.");
        }
        _vertical = setting;
        this.hasBeenReset = true;
      }
    },
    "snapToLines": {
      enumerable: true,
      get: function() {
        return _snapToLines;
      },
      set: function(value) {
        _snapToLines = !!value;
        this.hasBeenReset = true;
      }
    },
    "line": {
      enumerable: true,
      get: function() {
        return _line;
      },
      set: function(value) {
        if (typeof value !== "number" && value !== autoKeyword) {
          throw new SyntaxError("Line: an invalid number or illegal string was specified.");
        }
        _line = value;
        this.hasBeenReset = true;
      }
    },
    "lineAlign": {
      enumerable: true,
      get: function() {
        return _lineAlign;
      },
      set: function(value) {
        var setting = findAlignSetting(value);
        if (!setting) {
          console.warn("lineAlign: an invalid or illegal string was specified.");
        } else {
          _lineAlign = setting;
          this.hasBeenReset = true;
        }
      }
    },
    "position": {
      enumerable: true,
      get: function() {
        return _position;
      },
      set: function(value) {
        if (value < 0 || value > 100) {
          throw new Error("Position must be between 0 and 100.");
        }
        _position = value;
        this.hasBeenReset = true;
      }
    },
    "positionAlign": {
      enumerable: true,
      get: function() {
        return _positionAlign;
      },
      set: function(value) {
        var setting = findAlignSetting(value);
        if (!setting) {
          console.warn("positionAlign: an invalid or illegal string was specified.");
        } else {
          _positionAlign = setting;
          this.hasBeenReset = true;
        }
      }
    },
    "size": {
      enumerable: true,
      get: function() {
        return _size;
      },
      set: function(value) {
        if (value < 0 || value > 100) {
          throw new Error("Size must be between 0 and 100.");
        }
        _size = value;
        this.hasBeenReset = true;
      }
    },
    "align": {
      enumerable: true,
      get: function() {
        return _align;
      },
      set: function(value) {
        var setting = findAlignSetting(value);
        if (!setting) {
          throw new SyntaxError("align: an invalid or illegal alignment string was specified.");
        }
        _align = setting;
        this.hasBeenReset = true;
      }
    }
  });
  this.displayState = void 0;
}
VTTCue.prototype.getCueAsHTML = function() {
  return WebVTT.convertCueToDOMTree(window, this.text);
};
var vttcue = VTTCue;
var scrollSetting = {
  "": true,
  "up": true
};
function findScrollSetting(value) {
  if (typeof value !== "string") {
    return false;
  }
  var scroll = scrollSetting[value.toLowerCase()];
  return scroll ? value.toLowerCase() : false;
}
function isValidPercentValue(value) {
  return typeof value === "number" && (value >= 0 && value <= 100);
}
function VTTRegion() {
  var _width = 100;
  var _lines = 3;
  var _regionAnchorX = 0;
  var _regionAnchorY = 100;
  var _viewportAnchorX = 0;
  var _viewportAnchorY = 100;
  var _scroll = "";
  Object.defineProperties(this, {
    "width": {
      enumerable: true,
      get: function() {
        return _width;
      },
      set: function(value) {
        if (!isValidPercentValue(value)) {
          throw new Error("Width must be between 0 and 100.");
        }
        _width = value;
      }
    },
    "lines": {
      enumerable: true,
      get: function() {
        return _lines;
      },
      set: function(value) {
        if (typeof value !== "number") {
          throw new TypeError("Lines must be set to a number.");
        }
        _lines = value;
      }
    },
    "regionAnchorY": {
      enumerable: true,
      get: function() {
        return _regionAnchorY;
      },
      set: function(value) {
        if (!isValidPercentValue(value)) {
          throw new Error("RegionAnchorX must be between 0 and 100.");
        }
        _regionAnchorY = value;
      }
    },
    "regionAnchorX": {
      enumerable: true,
      get: function() {
        return _regionAnchorX;
      },
      set: function(value) {
        if (!isValidPercentValue(value)) {
          throw new Error("RegionAnchorY must be between 0 and 100.");
        }
        _regionAnchorX = value;
      }
    },
    "viewportAnchorY": {
      enumerable: true,
      get: function() {
        return _viewportAnchorY;
      },
      set: function(value) {
        if (!isValidPercentValue(value)) {
          throw new Error("ViewportAnchorY must be between 0 and 100.");
        }
        _viewportAnchorY = value;
      }
    },
    "viewportAnchorX": {
      enumerable: true,
      get: function() {
        return _viewportAnchorX;
      },
      set: function(value) {
        if (!isValidPercentValue(value)) {
          throw new Error("ViewportAnchorX must be between 0 and 100.");
        }
        _viewportAnchorX = value;
      }
    },
    "scroll": {
      enumerable: true,
      get: function() {
        return _scroll;
      },
      set: function(value) {
        var setting = findScrollSetting(value);
        if (setting === false) {
          console.warn("Scroll: an invalid or illegal string was specified.");
        } else {
          _scroll = setting;
        }
      }
    }
  });
}
var vttregion = VTTRegion;
var window$1 = window_1;
var vttjs = browserIndex.exports = {
  WebVTT: vtt,
  VTTCue: vttcue,
  VTTRegion: vttregion
};
window$1.vttjs = vttjs;
window$1.WebVTT = vttjs.WebVTT;
var cueShim = vttjs.VTTCue;
var regionShim = vttjs.VTTRegion;
var nativeVTTCue = window$1.VTTCue;
var nativeVTTRegion = window$1.VTTRegion;
vttjs.shim = function() {
  window$1.VTTCue = cueShim;
  window$1.VTTRegion = regionShim;
};
vttjs.restore = function() {
  window$1.VTTCue = nativeVTTCue;
  window$1.VTTRegion = nativeVTTRegion;
};
if (!window$1.VTTCue) {
  vttjs.shim();
}
export {
  browserIndex as b
};
